/*____________________________________________________________________________
	Copyright (C) 2000 Networks Associates Technology, Inc.
	All rights reserved.


	$Id: pgpWipeDeleteNT.c,v 1.7 2000/10/06 14:04:23 wjb Exp $
____________________________________________________________________________*/
#include "ntddk.h"

#include "PGPsdkDriver.h"
#include "pgpDriverNT.h"

#include "pgpWipeDeleteNT.h"

//----------------------------------------------------------------------
//                         DEFINES
//----------------------------------------------------------------------

//
// NT Final Build number
//
#define NT4FINAL        1381

//----------------------------------------------------------------------
//                         TYPEDEFS
//----------------------------------------------------------------------

//
// This is passed to the CLOSE completion routine for an PGPWDNT
//
typedef struct {
	PIRP					Irp;

#if	(_WIN32_WINNT >= 0x0500)
	PIO_WORKITEM			WorkItem;
#else	// _WIN32_WINNT < 0x5000
	WORK_QUEUE_ITEM			WorkItem;
#endif	// _WIN32_WINNT >= 0x5000

	PVOID					Token;
	PDEVICE_OBJECT			DeviceObject;
	char *					FileName;
} PGPWDNT_COMPLETE_CONTEXT, *PPGPWDNT_COMPLETE_CONTEXT;


//----------------------------------------------------------------------
//                         FORWARD DEFINES
//----------------------------------------------------------------------

//
// File name information
//

#if	(_WIN32_WINNT < 0x0500)

typedef struct _FILE_NAME_INFORMATION {
    ULONG FileNameLength;
    WCHAR FileName[1];
} FILE_NAME_INFORMATION, *PFILE_NAME_INFORMATION;

#endif	// _WIN32_WINNT < 0x5000

typedef struct _FILE_FS_ATTRIBUTE_INFORMATION {
    ULONG FileSystemAttributes;
    LONG MaximumComponentNameLength;
    ULONG FileSystemNameLength;
    WCHAR FileSystemName[1];
} FILE_FS_ATTRIBUTE_INFORMATION, *PFILE_FS_ATTRIBUTE_INFORMATION;


// prototypes for external functions
int __cdecl swprintf( WCHAR *, const WCHAR *, ... );

HANDLE
PsReferencePrimaryToken(
		PEPROCESS Process );

NTSTATUS NTAPI
ZwQueryVolumeInformationFile(
		HANDLE FileHandle,
		PIO_STATUS_BLOCK IoStatusBlock,
		PVOID FsInformation,
		ULONG Length,
		FS_INFORMATION_CLASS FsInformationClass );

//
// These are prototypes for PGPWDNT's Fast I/O hooks. The originals
// prototypes can be found in NTDDK.H
//
BOOLEAN  PGPWDNTFastIoCheckifPossible(
				IN PFILE_OBJECT FileObject,
				IN PLARGE_INTEGER FileOffset,
				IN ULONG Length,
				IN BOOLEAN Wait,
				IN ULONG LockKey,
				IN BOOLEAN CheckForReadOperation,
				OUT PIO_STATUS_BLOCK IoStatus,
				IN PDEVICE_OBJECT DeviceObject );
BOOLEAN  PGPWDNTFastIoRead(
				IN PFILE_OBJECT FileObject,
				IN PLARGE_INTEGER FileOffset,
				IN ULONG Length,
				IN BOOLEAN Wait,
				IN ULONG LockKey,
				OUT PVOID Buffer,
				OUT PIO_STATUS_BLOCK IoStatus,
				IN PDEVICE_OBJECT DeviceObject );
BOOLEAN  PGPWDNTFastIoWrite(
				IN PFILE_OBJECT FileObject,
				IN PLARGE_INTEGER FileOffset,
				IN ULONG Length,
				IN BOOLEAN Wait,
				IN ULONG LockKey,
				IN PVOID Buffer,
				OUT PIO_STATUS_BLOCK IoStatus,
				IN PDEVICE_OBJECT DeviceObject );
BOOLEAN  PGPWDNTFastIoQueryBasicInfo(
				IN PFILE_OBJECT FileObject,
				IN BOOLEAN Wait,
				OUT PFILE_BASIC_INFORMATION Buffer,
				OUT PIO_STATUS_BLOCK IoStatus,
				IN PDEVICE_OBJECT DeviceObject );
BOOLEAN  PGPWDNTFastIoQueryStandardInfo(
				IN PFILE_OBJECT FileObject,
				IN BOOLEAN Wait,
				OUT PFILE_STANDARD_INFORMATION Buffer,
				OUT PIO_STATUS_BLOCK IoStatus,
				IN PDEVICE_OBJECT DeviceObject );
BOOLEAN  PGPWDNTFastIoLock(
				IN PFILE_OBJECT FileObject,
				IN PLARGE_INTEGER FileOffset,
				IN PLARGE_INTEGER Length,
				PEPROCESS ProcessId,
				ULONG Key,
				BOOLEAN FailImmediately,
				BOOLEAN ExclusiveLock,
				OUT PIO_STATUS_BLOCK IoStatus,
				IN PDEVICE_OBJECT DeviceObject );
BOOLEAN  PGPWDNTFastIoUnlockSingle(
				IN PFILE_OBJECT FileObject,
				IN PLARGE_INTEGER FileOffset,
				IN PLARGE_INTEGER Length,
				PEPROCESS ProcessId,
				ULONG Key,
				OUT PIO_STATUS_BLOCK IoStatus,
				IN PDEVICE_OBJECT DeviceObject );
BOOLEAN  PGPWDNTFastIoUnlockAll(
				IN PFILE_OBJECT FileObject,
				PEPROCESS ProcessId,
				OUT PIO_STATUS_BLOCK IoStatus,
				IN PDEVICE_OBJECT DeviceObject );
BOOLEAN  PGPWDNTFastIoUnlockAllByKey(
				IN PFILE_OBJECT FileObject,
				PEPROCESS ProcessId,
				ULONG Key,
				OUT PIO_STATUS_BLOCK IoStatus,
				IN PDEVICE_OBJECT DeviceObject );
BOOLEAN  PGPWDNTFastIoDeviceControl(
				IN PFILE_OBJECT FileObject,
				IN BOOLEAN Wait,
				IN PVOID InputBuffer,
				IN ULONG InputBufferLength,
				OUT PVOID OutbufBuffer,
				IN ULONG OutputBufferLength,
				IN ULONG IoControlCode,
				OUT PIO_STATUS_BLOCK IoStatus,
				IN PDEVICE_OBJECT DeviceObject );
VOID     PGPWDNTFastIoAcquireFile(
				PFILE_OBJECT FileObject );
VOID     PGPWDNTFastIoReleaseFile(
				PFILE_OBJECT FileObject );
VOID     PGPWDNTFastIoDetachDevice(
				PDEVICE_OBJECT SourceDevice,
				PDEVICE_OBJECT TargetDevice );

//
// These are new NT 4.0 Fast I/O calls
//
BOOLEAN  PGPWDNTFastIoQueryNetworkOpenInfo(
				IN PFILE_OBJECT FileObject,
				IN BOOLEAN Wait,
				OUT struct _FILE_NETWORK_OPEN_INFORMATION *Buffer,
				OUT struct _IO_STATUS_BLOCK *IoStatus,
				IN PDEVICE_OBJECT DeviceObject );
NTSTATUS PGPWDNTFastIoAcquireForModWrite(
				IN PFILE_OBJECT FileObject,
				IN PLARGE_INTEGER EndingOffset,
				OUT struct _ERESOURCE **ResourceToRelease,
				IN PDEVICE_OBJECT DeviceObject );
BOOLEAN  PGPWDNTFastIoMdlRead(
				IN PFILE_OBJECT FileObject,
				IN PLARGE_INTEGER FileOffset,
				IN ULONG Length,
				IN ULONG LockKey,
				OUT PMDL *MdlChain,
				OUT PIO_STATUS_BLOCK IoStatus,
				IN PDEVICE_OBJECT DeviceObject );
BOOLEAN  PGPWDNTFastIoMdlReadComplete(
				IN PFILE_OBJECT FileObject,
				IN PMDL MdlChain,
				IN PDEVICE_OBJECT DeviceObject );
BOOLEAN  PGPWDNTFastIoPrepareMdlWrite(
				IN PFILE_OBJECT FileObject,
				IN PLARGE_INTEGER FileOffset,
				IN ULONG Length,
				IN ULONG LockKey,
				OUT PMDL *MdlChain,
				OUT PIO_STATUS_BLOCK IoStatus,
				IN PDEVICE_OBJECT DeviceObject );
BOOLEAN  PGPWDNTFastIoMdlWriteComplete(
				IN PFILE_OBJECT FileObject,
				IN PLARGE_INTEGER FileOffset,
				IN PMDL MdlChain,
				IN PDEVICE_OBJECT DeviceObject );
BOOLEAN  PGPWDNTFastIoReadCompressed(
				IN PFILE_OBJECT FileObject,
				IN PLARGE_INTEGER FileOffset,
				IN ULONG Length,
				IN ULONG LockKey,
				OUT PVOID Buffer,
				OUT PMDL *MdlChain,
				OUT PIO_STATUS_BLOCK IoStatus,
				OUT struct _COMPRESSED_DATA_INFO *CompressedDataInfo,
				IN ULONG CompressedDataInfoLength,
				IN PDEVICE_OBJECT DeviceObject );
BOOLEAN  PGPWDNTFastIoWriteCompressed(
				IN PFILE_OBJECT FileObject,
				IN PLARGE_INTEGER FileOffset,
				IN ULONG Length,
				IN ULONG LockKey,
				IN PVOID Buffer,
				OUT PMDL *MdlChain,
				OUT PIO_STATUS_BLOCK IoStatus,
				IN struct _COMPRESSED_DATA_INFO *CompressedDataInfo,
				IN ULONG CompressedDataInfoLength,
				IN PDEVICE_OBJECT DeviceObject );
BOOLEAN  PGPWDNTFastIoMdlReadCompleteCompressed(
				IN PFILE_OBJECT FileObject,
				IN PMDL MdlChain,
				IN PDEVICE_OBJECT DeviceObject );
BOOLEAN  PGPWDNTFastIoMdlWriteCompleteCompressed(
				IN PFILE_OBJECT FileObject,
				IN PLARGE_INTEGER FileOffset,
				IN PMDL MdlChain,
				IN PDEVICE_OBJECT DeviceObject );
BOOLEAN  PGPWDNTFastIoQueryOpen(
				IN struct _IRP *Irp,
				OUT PFILE_NETWORK_OPEN_INFORMATION NetworkInformation,
				IN PDEVICE_OBJECT DeviceObject );
NTSTATUS PGPWDNTFastIoReleaseForModWrite(
				IN PFILE_OBJECT FileObject,
				IN struct _ERESOURCE *ResourceToRelease,
				IN PDEVICE_OBJECT DeviceObject );
NTSTATUS PGPWDNTFastIoAcquireForCcFlush(
				IN PFILE_OBJECT FileObject,
				IN PDEVICE_OBJECT DeviceObject );
NTSTATUS PGPWDNTFastIoReleaseForCcFlush(
				IN PFILE_OBJECT FileObject,
				IN PDEVICE_OBJECT DeviceObject );

//----------------------------------------------------------------------
//                         GLOBALS
//----------------------------------------------------------------------

//
// This is PGPWDNT's user-inteface device object. It is addressed
// by calls from the GUI including CreateFile and DeviceIoControl
//
PDEVICE_OBJECT		GUIDevice;

//
// Table of our hook devices for each drive letter. This makes it
// easy to look up the device object that was created to hook a
// particular drive.
//
PDEVICE_OBJECT		LDriveDevices[26];

//
// System process
//
PEPROCESS			SystemProcess;

ULONG				tc_callback		=	0;
PKTHREAD			tc_thread		=	0;
ULONG				tc_flags		=	0;

//
// Undocumented ntoskrnl variable
//
extern PSHORT		NtBuildNumber;

//
// Local globals
//
KSEMAPHORE			apc_sem;
int					num_scheds;

//
// This PGPWDNT's Fast I/O dispatch table. Note that NT assumes that
// file system drivers support some Fast I/O calls, so this table must
// be present for an file system filter driver
//
FAST_IO_DISPATCH    FastIOHook = {
    sizeof(FAST_IO_DISPATCH),
    PGPWDNTFastIoCheckifPossible,
    PGPWDNTFastIoRead,
    PGPWDNTFastIoWrite,
    PGPWDNTFastIoQueryBasicInfo,
    PGPWDNTFastIoQueryStandardInfo,
    PGPWDNTFastIoLock,
    PGPWDNTFastIoUnlockSingle,
    PGPWDNTFastIoUnlockAll,
    PGPWDNTFastIoUnlockAllByKey,
    PGPWDNTFastIoDeviceControl,
    PGPWDNTFastIoAcquireFile,
    PGPWDNTFastIoReleaseFile,
    PGPWDNTFastIoDetachDevice,

    //
    // new for NT 4.0
    //
    PGPWDNTFastIoQueryNetworkOpenInfo,
    PGPWDNTFastIoAcquireForModWrite,
    PGPWDNTFastIoMdlRead,
    PGPWDNTFastIoMdlReadComplete,
    PGPWDNTFastIoPrepareMdlWrite,
    PGPWDNTFastIoMdlWriteComplete,
    PGPWDNTFastIoReadCompressed,
    PGPWDNTFastIoWriteCompressed,
    PGPWDNTFastIoMdlReadCompleteCompressed,
    PGPWDNTFastIoMdlWriteCompleteCompressed,
    PGPWDNTFastIoQueryOpen,
    PGPWDNTFastIoReleaseForModWrite,
    PGPWDNTFastIoAcquireForCcFlush,
    PGPWDNTFastIoReleaseForCcFlush
};



//----------------------------------------------------------------------
//             I R P    R E L A T E D   F U N C T I O N S
//----------------------------------------------------------------------

//----------------------------------------------------------------------
//
// PGPWDNTIoComplete
//
// This routine is used to handle I/O (read OR write) completion.
//
//----------------------------------------------------------------------
static NTSTATUS PGPWDNTIoComplete(PDEVICE_OBJECT DeviceObject,
				    PIRP Irp,
				    PVOID Context)
{
    //
    // Copy the status information back into the "user" IOSB.
    //
    *Irp->UserIosb = Irp->IoStatus;

    if( !NT_SUCCESS(Irp->IoStatus.Status) ) {

        PGPdbgPrint(("   ERROR ON IRP: %x\n", Irp->IoStatus.Status ));
    }

    //
    // Set the user event - wakes up the mainline code doing this.
    //
    KeSetEvent(Irp->UserEvent, 0, FALSE);

    //
    // Free the IRP now that we are done with it.
    //
    IoFreeIrp(Irp);

    //
    // We return STATUS_MORE_PROCESSING_REQUIRED because this "magic" return
	// value tells the I/O Manager that additional processing will be done by
	// this driver to the IRP - in fact, it might (as it is in this case)
	// already BE done - and the IRP cannot be completed.
    //
    return STATUS_MORE_PROCESSING_REQUIRED;
}


//----------------------------------------------------------------------
//
// PGPWDNTIsDirectory
//
// Returns TRUE if its a directory or we can't tell, FALSE if its a file.
//
//----------------------------------------------------------------------
BOOLEAN PGPWDNTIsDirectory(PDEVICE_OBJECT DeviceObject,
                            PFILE_OBJECT FileObject )
{
    PIRP irp;
    KEVENT event;
    IO_STATUS_BLOCK IoStatusBlock;
    PIO_STACK_LOCATION ioStackLocation;
    FILE_STANDARD_INFORMATION fileInfo;

    //
    // First, start by initializing the event
    //
    KeInitializeEvent(&event, SynchronizationEvent, FALSE);

    //
    // Allocate an irp for this request.  This could also come from a
    // private pool, for instance.
    //
    irp = IoAllocateIrp(DeviceObject->StackSize, FALSE);

    if (!irp) {

        //
        // Failure!
        //
        return FALSE;
    }

    irp->AssociatedIrp.SystemBuffer = &fileInfo;
    irp->UserEvent = &event;
    irp->UserIosb = &IoStatusBlock;
    irp->Tail.Overlay.Thread = PsGetCurrentThread();
    irp->Tail.Overlay.OriginalFileObject = FileObject;
    irp->RequestorMode = KernelMode;
    irp->Flags = 0;

    //
    // Set up the I/O stack location.
    //
    ioStackLocation = IoGetNextIrpStackLocation(irp);
    ioStackLocation->MajorFunction = IRP_MJ_QUERY_INFORMATION;
    ioStackLocation->DeviceObject = DeviceObject;
    ioStackLocation->FileObject = FileObject;
    ioStackLocation->Parameters.QueryVolume.Length =
								sizeof(FILE_STANDARD_INFORMATION);
    ioStackLocation->Parameters.QueryVolume.FsInformationClass =
								FileStandardInformation;

    //
    // Set the completion routine.
    //
    IoSetCompletionRoutine(irp, PGPWDNTIoComplete, 0, TRUE, TRUE, TRUE);

    //
    // Send the request to the lower layer driver.
    //
    (void) IoCallDriver(DeviceObject, irp);

    //
    // Wait for the I/O
    //
    KeWaitForSingleObject(&event, Executive, KernelMode, TRUE, 0);

    //
    // Return whether its a directory or not
    //
    if( !NT_SUCCESS( irp->IoStatus.Status ) || fileInfo.Directory) {

        return TRUE;

    } else {

        return FALSE;
    }
}


//----------------------------------------------------------------------
//
// PGPWDNTGetFileName
//
// This function retrieves the "standard" information for the
// underlying file system, asking for the filename in particular.
//
//----------------------------------------------------------------------
BOOLEAN PGPWDNTGetFileName(BOOLEAN IsNTFS,
                            PDEVICE_OBJECT DeviceObject,
                            PFILE_OBJECT FileObject,
                            PUCHAR FileName, ULONG FileNameLength )
{
    PIRP irp;
    KEVENT event;
    IO_STATUS_BLOCK IoStatusBlock;
    PIO_STACK_LOCATION ioStackLocation;
    PVOID fsContext2;

    PGPdbgVerbosePrint(("Getting file name for %x\n", FileObject));

    //
    // Initialize the event
    //
    KeInitializeEvent(&event, SynchronizationEvent, FALSE);

    //
    // Allocate an irp for this request.  This could also come from a
    // private pool, for instance.
    //
    irp = IoAllocateIrp(DeviceObject->StackSize, FALSE);

    if (!irp) {

        //
        // Failure!
        //
        return FALSE;
    }

    //
    // Zap Fscontext2 (the CCB) so that NTFS will give us the long name
    //
    if( IsNTFS ) {

        fsContext2 = FileObject->FsContext2;

        FileObject->FsContext2 = NULL;
    }

    irp->AssociatedIrp.SystemBuffer = FileName;
    irp->UserEvent = &event;
    irp->UserIosb = &IoStatusBlock;
    irp->Tail.Overlay.Thread = PsGetCurrentThread();
    irp->Tail.Overlay.OriginalFileObject = FileObject;
    irp->RequestorMode = KernelMode;
    irp->Flags = 0;

    //
    // Set up the I/O stack location.
    //
    ioStackLocation = IoGetNextIrpStackLocation(irp);
    ioStackLocation->MajorFunction = IRP_MJ_QUERY_INFORMATION;
    ioStackLocation->DeviceObject = DeviceObject;
    ioStackLocation->FileObject = FileObject;
    ioStackLocation->Parameters.QueryFile.Length = FileNameLength;
    ioStackLocation->Parameters.QueryFile.FileInformationClass =
							FileNameInformation;

    //
    // Set the completion routine.
    //
    IoSetCompletionRoutine(irp, PGPWDNTIoComplete, 0, TRUE, TRUE, TRUE);

    //
    // Send it to the FSD
    //
    (void) IoCallDriver(DeviceObject, irp);

    //
    // Wait for the I/O
    //
    KeWaitForSingleObject(&event, Executive, KernelMode, TRUE, 0);

    //
    // Restore the fscontext
    //
    if( IsNTFS ) {

        FileObject->FsContext2 = fsContext2;
    }

    //
    // Done!
    //
    return( NT_SUCCESS( irp->IoStatus.Status ));
}

//----------------------------------------------------------------------
//
// PGPWDNTSetDispositionFile
//
// Changes the delete status on a file
//
//----------------------------------------------------------------------
BOOLEAN PGPWDNTSetDispositionFile(PDEVICE_OBJECT DeviceObject,
			           PFILE_OBJECT FileObject, BOOLEAN Delete )
{
    PIRP irp;
    KEVENT event;
    IO_STATUS_BLOCK IoStatusBlock;
    PIO_STACK_LOCATION ioStackLocation;
    FILE_DISPOSITION_INFORMATION disposition;

    //
    // Change the delete status
    //
    disposition.DeleteFile = Delete;

    //
    // Initialize the event
    //
    KeInitializeEvent(&event, SynchronizationEvent, FALSE);

    //
    // Allocate an irp for this request.  This could also come from a
    // private pool, for instance.
    //
    irp = IoAllocateIrp(DeviceObject->StackSize, FALSE);

    if (!irp) {

        //
        // Failure!
        //
        return FALSE;
    }

    irp->AssociatedIrp.SystemBuffer = &disposition;
    irp->UserEvent = &event;
    irp->UserIosb = &IoStatusBlock;
    irp->Tail.Overlay.Thread = PsGetCurrentThread();
    irp->Tail.Overlay.OriginalFileObject = FileObject;
    irp->RequestorMode = KernelMode;
    irp->Flags = 0;

    //
    // Set up the I/O stack location.
    //
    ioStackLocation = IoGetNextIrpStackLocation(irp);
    ioStackLocation->MajorFunction = IRP_MJ_SET_INFORMATION;
    ioStackLocation->DeviceObject = DeviceObject;
    ioStackLocation->FileObject = FileObject;
    ioStackLocation->Parameters.SetFile.FileInformationClass =
							FileDispositionInformation;

    //
    // Set the completion routine.
    //
    IoSetCompletionRoutine(irp, PGPWDNTIoComplete, 0, TRUE, TRUE, TRUE);

    //
    // Send it to the FSD
    //
    (void) IoCallDriver(DeviceObject, irp);

    //
    // Wait for the I/O
    //
    KeWaitForSingleObject(&event, Executive, KernelMode, TRUE, 0);

    //
    // Done!
    //
    return TRUE;

}


//----------------------------------------------------------------------
//                F A S T I O   R O U T I N E S
//----------------------------------------------------------------------

//----------------------------------------------------------------------
//
// PGPWDNTFastIoCheckIfPossible
//
//----------------------------------------------------------------------
BOOLEAN
PGPWDNTFastIoCheckifPossible(
		IN PFILE_OBJECT FileObject,
		IN PLARGE_INTEGER FileOffset,
		IN ULONG Length,
		IN BOOLEAN Wait,
		IN ULONG LockKey,
		IN BOOLEAN CheckForReadOperation,
		OUT PIO_STATUS_BLOCK IoStatus,
		IN PDEVICE_OBJECT DeviceObject )
{
    BOOLEAN				retval = FALSE;
    PDEVEXTENSION		pdeve;

    pdeve = DeviceObject->DeviceExtension;

	// if not intended for us, force OS to create IRP
	if (pdeve->ulDeviceType != WIPE_DELETE_DEV)
		return FALSE;

    if( pdeve->pdevoNext->DriverObject->
					FastIoDispatch->FastIoCheckIfPossible )
	{
        retval = pdeve->pdevoNext->DriverObject->
					FastIoDispatch->FastIoCheckIfPossible(
							FileObject, FileOffset, Length,
							Wait, LockKey, CheckForReadOperation,
							IoStatus, pdeve->pdevoNext );
    }
    return retval;
}


//----------------------------------------------------------------------
//
// PGPWDNTFastIoRead
//
//----------------------------------------------------------------------
BOOLEAN
PGPWDNTFastIoRead(
		IN PFILE_OBJECT FileObject,
		IN PLARGE_INTEGER FileOffset,
		IN ULONG Length,
		IN BOOLEAN Wait,
		IN ULONG LockKey,
		OUT PVOID Buffer,
		OUT PIO_STATUS_BLOCK IoStatus,
		IN PDEVICE_OBJECT DeviceObject )
{
    BOOLEAN             retval = FALSE;
    PDEVEXTENSION		pdeve;

    pdeve = DeviceObject->DeviceExtension;

	// if not intended for us, force OS to create IRP
	if (pdeve->ulDeviceType != WIPE_DELETE_DEV)
		return FALSE;

    if( pdeve->pdevoNext->DriverObject->FastIoDispatch->FastIoRead )
	{
        retval = pdeve->pdevoNext->DriverObject->FastIoDispatch->FastIoRead(
            FileObject, FileOffset, Length,
            Wait, LockKey, Buffer, IoStatus, pdeve->pdevoNext );
    }
    return retval;
}


//----------------------------------------------------------------------
//
// PGPWDNTFastIoWrite
//
//----------------------------------------------------------------------
BOOLEAN
PGPWDNTFastIoWrite(
		IN PFILE_OBJECT FileObject,
		IN PLARGE_INTEGER FileOffset,
		IN ULONG Length,
		IN BOOLEAN Wait,
		IN ULONG LockKey,
		IN PVOID Buffer,
		OUT PIO_STATUS_BLOCK IoStatus,
		IN PDEVICE_OBJECT DeviceObject )
{
    BOOLEAN				retval = FALSE;
    PDEVEXTENSION		pdeve;

    pdeve = DeviceObject->DeviceExtension;

	// if not intended for us, force OS to create IRP
	if (pdeve->ulDeviceType != WIPE_DELETE_DEV)
		return FALSE;

    if( pdeve->pdevoNext->DriverObject->FastIoDispatch->FastIoWrite )
	{
        retval = pdeve->pdevoNext->
					DriverObject->FastIoDispatch->FastIoWrite(
							FileObject, FileOffset, Length, Wait, LockKey,
							Buffer, IoStatus, pdeve->pdevoNext );
    }
    return retval;
}


//----------------------------------------------------------------------
//
// PGPWDNTFastIoQueryBasicinfo
//
//----------------------------------------------------------------------
BOOLEAN
PGPWDNTFastIoQueryBasicInfo(
		IN PFILE_OBJECT FileObject,
		IN BOOLEAN Wait,
		OUT PFILE_BASIC_INFORMATION Buffer,
		OUT PIO_STATUS_BLOCK IoStatus,
		IN PDEVICE_OBJECT DeviceObject )
{
    BOOLEAN             retval = FALSE;
    PDEVEXTENSION		pdeve;

    pdeve = DeviceObject->DeviceExtension;

	// if not intended for us, force OS to create IRP
	if (pdeve->ulDeviceType != WIPE_DELETE_DEV)
		return FALSE;

    if( pdeve->pdevoNext->DriverObject->
					FastIoDispatch->FastIoQueryBasicInfo )
	{
        retval = pdeve->pdevoNext->DriverObject->
					FastIoDispatch->FastIoQueryBasicInfo(
							FileObject, Wait, Buffer,
							IoStatus, pdeve->pdevoNext );
    }
    return retval;
}


//----------------------------------------------------------------------
//
// PGPWDNTFastIoQueryStandardInfo
//
//----------------------------------------------------------------------
BOOLEAN
PGPWDNTFastIoQueryStandardInfo(
		IN PFILE_OBJECT FileObject,
		IN BOOLEAN Wait,
		OUT PFILE_STANDARD_INFORMATION Buffer,
		OUT PIO_STATUS_BLOCK IoStatus,
		IN PDEVICE_OBJECT DeviceObject )
{
    BOOLEAN             retval = FALSE;
    PDEVEXTENSION		pdeve;

    pdeve = DeviceObject->DeviceExtension;

	// if not intended for us, force OS to create IRP
	if (pdeve->ulDeviceType != WIPE_DELETE_DEV)
		return FALSE;

    if( pdeve->pdevoNext->DriverObject->
					FastIoDispatch->FastIoQueryStandardInfo )
	{
        retval = pdeve->pdevoNext->DriverObject->
					FastIoDispatch->FastIoQueryStandardInfo(
							FileObject, Wait, Buffer,
							IoStatus, pdeve->pdevoNext );
    }
    return retval;
}


//----------------------------------------------------------------------
//
// PGPWDNTFastIoLock
//
//----------------------------------------------------------------------
BOOLEAN
PGPWDNTFastIoLock(
		IN PFILE_OBJECT FileObject,
		IN PLARGE_INTEGER FileOffset,
		IN PLARGE_INTEGER Length,
		PEPROCESS ProcessId,
		ULONG Key,
		BOOLEAN FailImmediately,
		BOOLEAN ExclusiveLock,
		OUT PIO_STATUS_BLOCK IoStatus,
		IN PDEVICE_OBJECT DeviceObject )
{
    BOOLEAN             retval = FALSE;
    PDEVEXTENSION		pdeve;

    pdeve = DeviceObject->DeviceExtension;

	// if not intended for us, force OS to create IRP
	if (pdeve->ulDeviceType != WIPE_DELETE_DEV)
		return FALSE;

    if( pdeve->pdevoNext->DriverObject->FastIoDispatch->FastIoLock )
	{
        retval = pdeve->pdevoNext->DriverObject->FastIoDispatch->FastIoLock(
            FileObject, FileOffset, Length, ProcessId, Key, FailImmediately,
            ExclusiveLock, IoStatus, pdeve->pdevoNext );
    }
    return retval;
}


//----------------------------------------------------------------------
//
// PGPWDNTFastIoUnlockSingle
//
//----------------------------------------------------------------------
BOOLEAN
PGPWDNTFastIoUnlockSingle(
		IN PFILE_OBJECT FileObject,
		IN PLARGE_INTEGER FileOffset,
		IN PLARGE_INTEGER Length,
		PEPROCESS ProcessId,
		ULONG Key,
		OUT PIO_STATUS_BLOCK IoStatus,
		IN PDEVICE_OBJECT DeviceObject )
{
    BOOLEAN             retval = FALSE;
    PDEVEXTENSION		pdeve;

    pdeve = DeviceObject->DeviceExtension;

	// if not intended for us, force OS to create IRP
	if (pdeve->ulDeviceType != WIPE_DELETE_DEV)
		return FALSE;

    if( pdeve->pdevoNext->DriverObject->FastIoDispatch->FastIoUnlockSingle )
	{
        retval = pdeve->pdevoNext->
					DriverObject->FastIoDispatch->FastIoUnlockSingle(
							FileObject, FileOffset, Length, ProcessId, Key,
							IoStatus, pdeve->pdevoNext );
    }
    return retval;
}


//----------------------------------------------------------------------
//
// PGPWDNTFastIoUnlockAll
//
//----------------------------------------------------------------------
BOOLEAN
PGPWDNTFastIoUnlockAll(
		IN PFILE_OBJECT FileObject,
		PEPROCESS ProcessId,
		OUT PIO_STATUS_BLOCK IoStatus,
		IN PDEVICE_OBJECT DeviceObject )
{
    BOOLEAN             retval = FALSE;
    PDEVEXTENSION		pdeve;

    pdeve = DeviceObject->DeviceExtension;

	// if not intended for us, force OS to create IRP
	if (pdeve->ulDeviceType != WIPE_DELETE_DEV)
		return FALSE;

    if ( pdeve->pdevoNext->DriverObject->FastIoDispatch->FastIoUnlockAll )
	{
        retval = pdeve->pdevoNext->
					DriverObject->FastIoDispatch->FastIoUnlockAll(
							FileObject, ProcessId,
							IoStatus, pdeve->pdevoNext );
    }
    return retval;
}


//----------------------------------------------------------------------
//
// PGPWDNTFastIoUnlockAllByKey
//
//----------------------------------------------------------------------
BOOLEAN
PGPWDNTFastIoUnlockAllByKey(
		IN PFILE_OBJECT FileObject,
		PEPROCESS ProcessId,
		ULONG Key,
		OUT PIO_STATUS_BLOCK IoStatus,
		IN PDEVICE_OBJECT DeviceObject )
{
    BOOLEAN             retval = FALSE;
    PDEVEXTENSION		pdeve;

    pdeve = DeviceObject->DeviceExtension;

	// if not intended for us, force OS to create IRP
	if (pdeve->ulDeviceType != WIPE_DELETE_DEV)
		return FALSE;

    if( pdeve->pdevoNext->DriverObject->
					FastIoDispatch->FastIoUnlockAllByKey )
	{
        retval = pdeve->pdevoNext->DriverObject->
					FastIoDispatch->FastIoUnlockAllByKey(
							FileObject, ProcessId, Key,
							IoStatus, pdeve->pdevoNext );
    }
    return retval;
}


//----------------------------------------------------------------------
//
// PGPWDNTFastIoQueryNetworkOpenInfo
//
//----------------------------------------------------------------------
BOOLEAN
PGPWDNTFastIoQueryNetworkOpenInfo(
		IN PFILE_OBJECT FileObject,
		IN BOOLEAN Wait,
		OUT struct _FILE_NETWORK_OPEN_INFORMATION *Buffer,
		OUT PIO_STATUS_BLOCK IoStatus,
		IN PDEVICE_OBJECT DeviceObject )
{
    BOOLEAN             retval = FALSE;
    PDEVEXTENSION		pdeve;

    pdeve = DeviceObject->DeviceExtension;

	// if not intended for us, force OS to create IRP
	if (pdeve->ulDeviceType != WIPE_DELETE_DEV)
		return FALSE;

    if( pdeve->pdevoNext->DriverObject->
					FastIoDispatch->FastIoQueryNetworkOpenInfo )
	{
        retval = pdeve->pdevoNext->DriverObject->
					FastIoDispatch->FastIoQueryNetworkOpenInfo(
							FileObject, Wait, Buffer,
							IoStatus, pdeve->pdevoNext );
    }
    return retval;
}


//----------------------------------------------------------------------
//
// PGPWDNTFastIoAcquireForModWrite
//
//----------------------------------------------------------------------
NTSTATUS
PGPWDNTFastIoAcquireForModWrite(
		IN PFILE_OBJECT FileObject,
		IN PLARGE_INTEGER EndingOffset,
		OUT struct _ERESOURCE **ResourceToRelease,
		IN PDEVICE_OBJECT DeviceObject )
{
    NTSTATUS            retval = STATUS_NOT_SUPPORTED;
    PDEVEXTENSION		pdeve;

    pdeve = DeviceObject->DeviceExtension;

	// if not intended for us, force OS to create IRP
	if (pdeve->ulDeviceType != WIPE_DELETE_DEV)
		return FALSE;

    if( pdeve->pdevoNext->DriverObject->FastIoDispatch->AcquireForModWrite )
	{
        retval = pdeve->pdevoNext->DriverObject->
					FastIoDispatch->AcquireForModWrite(
							FileObject, EndingOffset,
							ResourceToRelease, pdeve->pdevoNext );
    }
    return retval;
}


//----------------------------------------------------------------------
//
// PGPWDNTFastIoMdlRead
//
//----------------------------------------------------------------------
BOOLEAN
PGPWDNTFastIoMdlRead(
		IN PFILE_OBJECT FileObject,
		IN PLARGE_INTEGER FileOffset,
		IN ULONG Length,
		IN ULONG LockKey,
		OUT PMDL *MdlChain,
		OUT PIO_STATUS_BLOCK IoStatus,
		IN PDEVICE_OBJECT DeviceObject )
{
    BOOLEAN             retval = FALSE;
    PDEVEXTENSION		pdeve;

    pdeve = DeviceObject->DeviceExtension;

	// if not intended for us, force OS to create IRP
	if (pdeve->ulDeviceType != WIPE_DELETE_DEV)
		return FALSE;

    if( pdeve->pdevoNext->DriverObject->FastIoDispatch->MdlRead )
	{
        retval = pdeve->pdevoNext->DriverObject->FastIoDispatch->MdlRead(
            FileObject, FileOffset, Length, LockKey, MdlChain,
            IoStatus, pdeve->pdevoNext );
    }
    return retval;
}


//----------------------------------------------------------------------
//
// PGPWDNTFastIoMdlReadComplete
//
//----------------------------------------------------------------------
BOOLEAN
PGPWDNTFastIoMdlReadComplete(
		IN PFILE_OBJECT FileObject,
		IN PMDL MdlChain,
		IN PDEVICE_OBJECT DeviceObject )
{
    BOOLEAN             retval = FALSE;
    PDEVEXTENSION		pdeve;

    pdeve = DeviceObject->DeviceExtension;

	// if not intended for us, force OS to create IRP
	if (pdeve->ulDeviceType != WIPE_DELETE_DEV)
		return FALSE;

    if( pdeve->pdevoNext->DriverObject->FastIoDispatch->MdlReadComplete )
	{
        retval = (BOOLEAN) pdeve->pdevoNext->DriverObject->
					FastIoDispatch->MdlReadComplete(
							FileObject, MdlChain, pdeve->pdevoNext );
    }
    return retval;
}


//----------------------------------------------------------------------
//
// PGPWDNTFastIoPrepareMdlWrite
//
//----------------------------------------------------------------------
BOOLEAN
PGPWDNTFastIoPrepareMdlWrite(
		IN PFILE_OBJECT FileObject,
		IN PLARGE_INTEGER FileOffset,
		IN ULONG Length,
		IN ULONG LockKey,
		OUT PMDL *MdlChain,
		OUT PIO_STATUS_BLOCK IoStatus,
		IN PDEVICE_OBJECT DeviceObject )
{
    BOOLEAN             retval = FALSE;
    PDEVEXTENSION		pdeve;

    pdeve = DeviceObject->DeviceExtension;

	// if not intended for us, force OS to create IRP
	if (pdeve->ulDeviceType != WIPE_DELETE_DEV)
		return FALSE;

    IoStatus->Status      = STATUS_NOT_SUPPORTED;
    IoStatus->Information = 0;

    if( pdeve->pdevoNext->DriverObject->FastIoDispatch->PrepareMdlWrite )
	{
        retval = pdeve->pdevoNext->DriverObject->
					FastIoDispatch->PrepareMdlWrite(
							FileObject, FileOffset, Length, LockKey,
							MdlChain, IoStatus, pdeve->pdevoNext );
    }
    return retval;
}


//----------------------------------------------------------------------
//
// PGPWDNTFastIoMdlWriteComplete
//
//----------------------------------------------------------------------
BOOLEAN
PGPWDNTFastIoMdlWriteComplete(
		IN PFILE_OBJECT FileObject,
		IN PLARGE_INTEGER FileOffset,
		IN PMDL MdlChain,
		IN PDEVICE_OBJECT DeviceObject )
{
    BOOLEAN             retval = FALSE;
    PDEVEXTENSION		pdeve;

    pdeve = DeviceObject->DeviceExtension;

	// if not intended for us, force OS to create IRP
	if (pdeve->ulDeviceType != WIPE_DELETE_DEV)
		return FALSE;

    if( pdeve->pdevoNext->DriverObject->FastIoDispatch->MdlWriteComplete )
	{
        retval = pdeve->pdevoNext->DriverObject->
					FastIoDispatch->MdlWriteComplete(
							FileObject, FileOffset,
							MdlChain, pdeve->pdevoNext );
    }
    return retval;
}


//----------------------------------------------------------------------
//
// PGPWDNTFastIoReadCompressed
//
//----------------------------------------------------------------------
BOOLEAN
PGPWDNTFastIoReadCompressed(
		IN PFILE_OBJECT FileObject,
		IN PLARGE_INTEGER FileOffset,
		IN ULONG Length,
		IN ULONG LockKey,
		OUT PVOID Buffer,
		OUT PMDL *MdlChain,
		OUT PIO_STATUS_BLOCK IoStatus,
		OUT struct _COMPRESSED_DATA_INFO *CompressedDataInfo,
		IN ULONG CompressedDataInfoLength,
		IN PDEVICE_OBJECT DeviceObject )
{
    BOOLEAN             retval = FALSE;
    PDEVEXTENSION		pdeve;

    pdeve = DeviceObject->DeviceExtension;

	// if not intended for us, force OS to create IRP
	if (pdeve->ulDeviceType != WIPE_DELETE_DEV)
		return FALSE;

    if( pdeve->pdevoNext->DriverObject->FastIoDispatch->FastIoReadCompressed )
	{
        retval = pdeve->pdevoNext->DriverObject->
					FastIoDispatch->FastIoReadCompressed(
							FileObject, FileOffset, Length, LockKey, Buffer,
							MdlChain, IoStatus, CompressedDataInfo,
							CompressedDataInfoLength, pdeve->pdevoNext );
    }
    return retval;
}


//----------------------------------------------------------------------
//
// PGPWDNTFastIoWriteCompressed
//
//----------------------------------------------------------------------
BOOLEAN
PGPWDNTFastIoWriteCompressed(
		IN PFILE_OBJECT FileObject,
		IN PLARGE_INTEGER FileOffset,
		IN ULONG Length,
		IN ULONG LockKey,
		OUT PVOID Buffer,
		OUT PMDL *MdlChain,
		OUT PIO_STATUS_BLOCK IoStatus,
		OUT struct _COMPRESSED_DATA_INFO *CompressedDataInfo,
		IN ULONG CompressedDataInfoLength,
		IN PDEVICE_OBJECT DeviceObject )
{
    BOOLEAN             retval = FALSE;
    PDEVEXTENSION		pdeve;

    pdeve = DeviceObject->DeviceExtension;

	// if not intended for us, force OS to create IRP
	if (pdeve->ulDeviceType != WIPE_DELETE_DEV)
		return FALSE;

    if( pdeve->pdevoNext->DriverObject->
					FastIoDispatch->FastIoWriteCompressed )
	{
        retval = pdeve->pdevoNext->DriverObject->
					FastIoDispatch->FastIoWriteCompressed(
							FileObject, FileOffset, Length, LockKey, Buffer,
							MdlChain, IoStatus, CompressedDataInfo,
							CompressedDataInfoLength, pdeve->pdevoNext );
    }
    return retval;
}


//----------------------------------------------------------------------
//
// PGPWDNTFastIoMdlReadCompleteCompressed
//
//----------------------------------------------------------------------
BOOLEAN
PGPWDNTFastIoMdlReadCompleteCompressed(
		IN PFILE_OBJECT FileObject,
		IN PMDL MdlChain,
		IN PDEVICE_OBJECT DeviceObject )
{
    BOOLEAN             retval = FALSE;
    PDEVEXTENSION		pdeve;

    pdeve = DeviceObject->DeviceExtension;

	// if not intended for us, force OS to create IRP
	if (pdeve->ulDeviceType != WIPE_DELETE_DEV)
		return FALSE;

    if( pdeve->pdevoNext->
					DriverObject->FastIoDispatch->MdlReadCompleteCompressed )
	{
        retval = pdeve->pdevoNext->DriverObject->
					FastIoDispatch->MdlReadCompleteCompressed(
							FileObject, MdlChain, pdeve->pdevoNext );
    }
    return retval;
}


//----------------------------------------------------------------------
//
// PGPWDNTFastIoMdlWriteCompleteCompressed
//
//----------------------------------------------------------------------
BOOLEAN
PGPWDNTFastIoMdlWriteCompleteCompressed(
		IN PFILE_OBJECT FileObject,
		IN PLARGE_INTEGER FileOffset,
		IN PMDL MdlChain,
		IN PDEVICE_OBJECT DeviceObject )
{
    BOOLEAN             retval = FALSE;
    PDEVEXTENSION		pdeve;

    pdeve = DeviceObject->DeviceExtension;

	// if not intended for us, force OS to create IRP
	if (pdeve->ulDeviceType != WIPE_DELETE_DEV)
		return FALSE;

    if( pdeve->pdevoNext->DriverObject->
					FastIoDispatch->MdlWriteCompleteCompressed )
	{
        retval = pdeve->pdevoNext->DriverObject->
					FastIoDispatch->MdlWriteCompleteCompressed(
							FileObject, FileOffset,
							MdlChain, pdeve->pdevoNext );
    }
    return retval;
}


//----------------------------------------------------------------------
//
// PGPWDNTFastIoQueryOpen
//
// This call actually passes an IRP!
//
//----------------------------------------------------------------------
BOOLEAN
PGPWDNTFastIoQueryOpen(
		IN PIRP Irp,
		OUT PFILE_NETWORK_OPEN_INFORMATION NetworkInformation,
		IN PDEVICE_OBJECT DeviceObject )
{
    BOOLEAN             retval = FALSE;
    PFILE_OBJECT        fileobject;
    PIO_STACK_LOCATION  currentIrpStack;
    PIO_STACK_LOCATION  nextIrpStack;
    PDEVEXTENSION		pdeve;

    pdeve = DeviceObject->DeviceExtension;

	// if not intended for us, force OS to create IRP
	if (pdeve->ulDeviceType != WIPE_DELETE_DEV)
		return FALSE;

    if( pdeve->pdevoNext->DriverObject->FastIoDispatch->FastIoQueryOpen )
	{
        currentIrpStack = IoGetCurrentIrpStackLocation(Irp);
        nextIrpStack    = IoGetNextIrpStackLocation(Irp);
        fileobject      = currentIrpStack->FileObject;

        //
        // copy parameters down to next level in the stack
        //
        *nextIrpStack = *currentIrpStack;
        nextIrpStack->DeviceObject = pdeve->pdevoNext;
        IoSetNextIrpStackLocation( Irp );

        retval = pdeve->pdevoNext->DriverObject->
					FastIoDispatch->FastIoQueryOpen(
							Irp, NetworkInformation, pdeve->pdevoNext );
    }
    return retval;
}


//----------------------------------------------------------------------
//
// PGPWDNTFastIoReleaseForModWrite
//
//----------------------------------------------------------------------
NTSTATUS
PGPWDNTFastIoReleaseForModWrite(
		IN PFILE_OBJECT FileObject,
		IN struct _ERESOURCE *ResourceToRelease,
		IN PDEVICE_OBJECT DeviceObject )
{
    NTSTATUS            retval = STATUS_NOT_SUPPORTED;
    PDEVEXTENSION		pdeve;

    pdeve = DeviceObject->DeviceExtension;

	// if not intended for us, force OS to create IRP
	if (pdeve->ulDeviceType != WIPE_DELETE_DEV)
		return FALSE;

    if( pdeve->pdevoNext->DriverObject->FastIoDispatch->ReleaseForModWrite )
	{
        retval = pdeve->pdevoNext->DriverObject->
					FastIoDispatch->ReleaseForModWrite(
							FileObject,  ResourceToRelease,
							pdeve->pdevoNext );
    }
    return retval;
}


//----------------------------------------------------------------------
//
// PGPWDNTFastIoAcquireForCcFlush
//
//----------------------------------------------------------------------
NTSTATUS
PGPWDNTFastIoAcquireForCcFlush(
		IN PFILE_OBJECT FileObject,
		IN PDEVICE_OBJECT DeviceObject )
{
    NTSTATUS            retval = STATUS_NOT_SUPPORTED;
    PDEVEXTENSION		pdeve;

    pdeve = DeviceObject->DeviceExtension;

	// if not intended for us, force OS to create IRP
	if (pdeve->ulDeviceType != WIPE_DELETE_DEV)
		return FALSE;

    if( pdeve->pdevoNext->DriverObject->FastIoDispatch->AcquireForCcFlush )
	{
        retval = pdeve->pdevoNext->DriverObject->
					FastIoDispatch->AcquireForCcFlush(
							FileObject, pdeve->pdevoNext );
    }
    return retval;
}


//----------------------------------------------------------------------
//
// PGPWDNTFastIoReleaseForCcFlush
//
//----------------------------------------------------------------------
NTSTATUS
PGPWDNTFastIoReleaseForCcFlush(
		IN PFILE_OBJECT FileObject,
		IN PDEVICE_OBJECT DeviceObject )
{
    NTSTATUS            retval = STATUS_NOT_SUPPORTED;
    PDEVEXTENSION		pdeve;

    pdeve = DeviceObject->DeviceExtension;

	// if not intended for us, force OS to create IRP
	if (pdeve->ulDeviceType != WIPE_DELETE_DEV)
		return FALSE;

    if( pdeve->pdevoNext->DriverObject->FastIoDispatch->ReleaseForCcFlush )
	{
        retval = pdeve->pdevoNext->DriverObject->
					FastIoDispatch->ReleaseForCcFlush(
							FileObject, pdeve->pdevoNext );
    }
    return retval;
}


//----------------------------------------------------------------------
//
// PGPWDNTFastIoDeviceControl
//
//----------------------------------------------------------------------
BOOLEAN
PGPWDNTFastIoDeviceControl(
		IN PFILE_OBJECT FileObject,
		IN BOOLEAN Wait,
		IN PVOID InputBuffer,
		IN ULONG InputBufferLength,
		OUT PVOID OutputBuffer,
		IN ULONG OutputBufferLength,
		IN ULONG IoControlCode,
		OUT PIO_STATUS_BLOCK IoStatus,
		IN PDEVICE_OBJECT DeviceObject )
{
    BOOLEAN             retval = FALSE;
    PDEVEXTENSION		pdeve;

    pdeve = DeviceObject->DeviceExtension;

	// if not intended for us, force OS to create IRP
	if (pdeve->ulDeviceType != WIPE_DELETE_DEV)
		return FALSE;

    //
    // Its a call for a file system, so pass it through
    //
    if( pdeve->pdevoNext->DriverObject->
					FastIoDispatch->FastIoDeviceControl )
	{
        retval = pdeve->pdevoNext->DriverObject->
					FastIoDispatch->FastIoDeviceControl(
							FileObject, Wait, InputBuffer, InputBufferLength,
							OutputBuffer, OutputBufferLength, IoControlCode,
							IoStatus, pdeve->pdevoNext );
    }

    return retval;
}


//----------------------------------------------------------------------
//
// PGPWDNTFastIoAcquireFile
//
//----------------------------------------------------------------------
VOID
PGPWDNTFastIoAcquireFile(
		PFILE_OBJECT FileObject )
{
    PDEVICE_OBJECT      deviceObject;
    PDEVEXTENSION		pdeve;

	deviceObject = IoGetRelatedDeviceObject( FileObject );
	pdeve = deviceObject->DeviceExtension;

	// if not intended for us, stop here
	if (pdeve->ulDeviceType != WIPE_DELETE_DEV)
		return;

    if( pdeve->pdevoNext->DriverObject->
					FastIoDispatch->AcquireFileForNtCreateSection )
	{
        pdeve->pdevoNext->DriverObject->
					FastIoDispatch->AcquireFileForNtCreateSection(
							FileObject );
    }
}


//----------------------------------------------------------------------
//
// PGPWDNTFastIoReleaseFile
//
//----------------------------------------------------------------------
VOID
PGPWDNTFastIoReleaseFile(
		PFILE_OBJECT FileObject )
{
    PDEVICE_OBJECT      deviceObject;
    PDEVEXTENSION		pdeve;

    deviceObject = IoGetRelatedDeviceObject( FileObject );
    pdeve = deviceObject->DeviceExtension;

	// if not intended for us, stop here
	if (pdeve->ulDeviceType != WIPE_DELETE_DEV)
		return;

    if( pdeve->pdevoNext->DriverObject->
				FastIoDispatch->ReleaseFileForNtCreateSection )
	{
        pdeve->pdevoNext->DriverObject->
				FastIoDispatch->ReleaseFileForNtCreateSection( FileObject );
    }
}


//----------------------------------------------------------------------
//
// PGPWDNTFastIoDetachDevice
//
// We get this call when a device that we have hooked is being deleted.
// This happens when, for example, a floppy is formatted. We have
// to detach from it and delete our device. We should notify the GUI
// that the hook state has changed, but its not worth the trouble.
//
//----------------------------------------------------------------------
VOID
PGPWDNTFastIoDetachDevice(
		PDEVICE_OBJECT SourceDevice,
		PDEVICE_OBJECT TargetDevice )
{
    PDEVEXTENSION		pdeve;
    ULONG               i;

	pdeve = SourceDevice->DeviceExtension;

	// if not intended for us, stop here
	if (pdeve->ulDeviceType != WIPE_DELETE_DEV)
		return;

	//
    // See if a device (like a floppy) is being removed out from under us.
	// If so, we have to detach from it before it disappears
    //
    for( i = 2; i < 26; i++ ) {

        if( SourceDevice == LDriveDevices[i] ) {

            //
            // We've hooked it, so we must detach
            //
            PGPdbgVerbosePrint(("PGPutil: Detaching from drive: %c\n",
                      pdeve->ucLogicalDrive ));

            IoDetachDevice( TargetDevice );
            IoDeleteDevice( SourceDevice );

            LDriveDevices[i] = NULL;

            return;
        }
    }

    //
    // Now we can pass the call through
    //
    if( pdeve->pdevoNext->DriverObject->
					FastIoDispatch->FastIoDetachDevice )
	{
        pdeve->pdevoNext->DriverObject->
					FastIoDispatch->FastIoDetachDevice(
            SourceDevice, TargetDevice );
    }
}

//----------------------------------------------------------------------
//
// PGPWDNTCheckFileForPGPWDNT
//
// See if this file is marked for delete. If so, get its name
// and see if this is a file that should be protected.
//
//----------------------------------------------------------------------
BOOLEAN
PGPWDNTCheckFileForPGPWDNT(
		PDEVICE_OBJECT DeviceObject,
		PIRP Irp,
		char **FileName)
{
    PIO_STACK_LOCATION      currentIrpStack = IoGetCurrentIrpStackLocation(Irp);
    PDEVEXTENSION			pdeve;
    PCHAR                   fileNameBuffer;
	UNICODE_STRING          fileNameUnicodeString;
	int i,length;
	char *name;

    //
    // Get the hook extension
    //
    pdeve = DeviceObject->DeviceExtension;

	// if not intended for us, force OS to create IRP
	if (pdeve->ulDeviceType != WIPE_DELETE_DEV)
		return FALSE;

	// We have nothing to do anyway....
	if(tc_callback==0)
		return FALSE;

    // First, make sure its not a directory
	if( PGPWDNTIsDirectory( pdeve->pdevoNext, currentIrpStack->FileObject ) )
	{
		return FALSE;
	}

    //
    // Get the name of the file.
    //
    fileNameBuffer = ExAllocatePool( NonPagedPool, 1024 );
    if( !PGPWDNTGetFileName( pdeve->bIsNTFS, pdeve->pdevoNext,
		currentIrpStack->FileObject,
		fileNameBuffer, 1024))
	{
        //
        // Couldn't get the name for some reason
        //
        ExFreePool( fileNameBuffer );
        return FALSE;
    }

    fileNameUnicodeString.Length =
		(USHORT) ((PFILE_NAME_INFORMATION) fileNameBuffer)->FileNameLength + 4;
    fileNameUnicodeString.MaximumLength = fileNameUnicodeString.Length;
    fileNameUnicodeString.Buffer =
		ExAllocatePool( PagedPool, fileNameUnicodeString.Length+2+4);
    swprintf( fileNameUnicodeString.Buffer, L"%C:", pdeve->ucLogicalDrive );
    RtlCopyMemory(	&fileNameUnicodeString.Buffer[2],
					((PFILE_NAME_INFORMATION) fileNameBuffer)->FileName,
					fileNameUnicodeString.Length);
    fileNameUnicodeString.Buffer[fileNameUnicodeString.Length/2] = 0;

    //
    // Determine the length of the name
    //
	i=0;
	length=0;

    while( fileNameUnicodeString.Buffer[i++] ) length++;

    //
    // First, make the ascii name
    //
    for( i = 0; i < length; i++ )
	{
		fileNameBuffer[i] = (CHAR) fileNameUnicodeString.Buffer[i];
    }
    fileNameBuffer[i] = 0;

	ExFreePool( fileNameUnicodeString.Buffer );

	name=strrchr(fileNameBuffer,'\\');

	if(name==NULL)
		name=fileNameBuffer;
	else
		name++;

	if(_strnicmp(name,"aaaaaaaa",8)==0)
	{
        // Its already been wiped by us. Go ahead and let OS delete it.
        PGPdbgVerbosePrint(("PGPutil: its already been wiped\n"));
        ExFreePool( fileNameBuffer );
        return FALSE;
    }

    *FileName  = fileNameBuffer;
    return TRUE;
}

/* The APC structure is defined in ntddk.h */

/* this is KERNEL_ROUTINE for our APC; in particular, it gets called when the APC
is being delivered. Usually one of predefined useful routines in the kernel are
used for this purpose, but we can't use any of them as none of them are exported. */

void MyRoutine(struct _KAPC *Apc,
               PKNORMAL_ROUTINE norm_routine,
               void *context,
               void *SysArg1,
               void *SysArg2)
{
    ExFreePool(Apc);
    return;
}

/* pointer to the APC we will create */

static struct _KAPC *apc;

/* KeInitializeApc() and KeInsertQueueApc() are the two functions needed to
send an APC; they're both exported but not prototyped in the DDK, so we
prototype them here. */

void KeInitializeApc(struct _KAPC *Apc, PKTHREAD thread,
                     unsigned char state_index,
                     PKKERNEL_ROUTINE ker_routine,
                     PKRUNDOWN_ROUTINE rd_routine,
                     PKNORMAL_ROUTINE nor_routine,
                     unsigned char mode,
                     void *context);


void KeInsertQueueApc(struct _KAPC *APC,
                      void *SysArg1,
                      void *SysArg2,
                      unsigned char arg4);

/* call this function when you need to send a user-mode APC to
the current thread. addr must be linear address of your user-mode
function to call:

void MyApcRoutine(ULONG arg1, ULONG arg2, ULONG arg3);
...
SendAddrToTheDriverUsingIoctl((ULONG)MyApcRoutine);

you should send it to the driver using your custom IOCTL.
arg1, arg2, arg3 are arbitrary ulong's which are passed to the function
residing at addr; this function should be prototyped as receiving three
parameters and returning void. */

void SendAPC(ULONG addr, PKTHREAD thread, ULONG arg1, ULONG arg2, ULONG arg3)
{
/* this is self-explanatory */

    apc=ExAllocatePool(NonPagedPool, sizeof(struct _KAPC));

/* Initialize the user-mode APC */

    KeInitializeApc(apc, thread, 0,
                    (PKKERNEL_ROUTINE)&MyRoutine, 0,
                    (PKNORMAL_ROUTINE)addr, 1, (PVOID)arg1);

/* Insert it to the queue of the target thread */

    KeInsertQueueApc(apc, (PVOID)arg2, (PVOID)arg3, 0);

/* Mark the current thread as alertable to force it to deliver the APC on
the next return to the user-mode. NOTE: severely undocumented code here!
*/

    *((unsigned char *)thread+0x4a)=1;
}

//----------------------------------------------------------------------
//
// PGPWDNTworkitem
//
// Called by the system, this is initiated in the completion callback.
// We use it to ensure we''re running at passive level when we do
// an PGPWDNT
//
//----------------------------------------------------------------------
#if	(_WIN32_WINNT >= 0x0500)
VOID PGPWDNTWorkItem( IN PDEVICE_OBJECT DeviceObject, PVOID Context )
#else	// _WIN32_WINNT < 0x5000
VOID PGPWDNTWorkItem( PVOID Context )
#endif	// _WIN32_WINNT >= 0x5000
{
	PPGPWDNT_COMPLETE_CONTEXT pwntcc;

	pwntcc=(PPGPWDNT_COMPLETE_CONTEXT)Context;

	num_scheds++;

    // Do the APC call back to the user callback
	SendAPC(tc_callback, tc_thread, (ULONG)pwntcc->FileName, 0, 0);

	KeWaitForSingleObject(&apc_sem,Executive, KernelMode, FALSE, NULL);

	// Free the filename
	ExFreePool(pwntcc->FileName);

	num_scheds--;

    // Complete the IRP
    pwntcc->Irp->IoStatus.Status = STATUS_SUCCESS;
    IoCompleteRequest(pwntcc->Irp,IO_NO_INCREMENT);

#if	(_WIN32_WINNT >= 0x0500)
	// free the work item
	IoFreeWorkItem(pwntcc->WorkItem);
#endif	// _WIN32_WINNT >= 0x5000

	// Free the context memory
	ExFreePool(pwntcc);
}

//----------------------------------------------------------------------
//     D I S P A T C H   A N D   H O O K   E N T R Y   P O I N T S
//----------------------------------------------------------------------


//----------------------------------------------------------------------
//
// PGPWDNTHookDone
//
// Gets control on the way back from a delete operation. At this point
// the file cannot be opened by anything else, so we can send it back
// to the user callback
//
//----------------------------------------------------------------------
NTSTATUS PGPWDNTHookDone( IN PDEVICE_OBJECT DeviceObject, IN PIRP Irp,
                           IN PVOID Context )
{
	PPGPWDNT_COMPLETE_CONTEXT pwntcc;

	pwntcc=(PPGPWDNT_COMPLETE_CONTEXT)Context;

    if( NT_SUCCESS(Irp->IoStatus.Status))
	{
#if	(_WIN32_WINNT >= 0x0500)

		pwntcc->WorkItem = IoAllocateWorkItem(DeviceObject);

		if (pwntcc->WorkItem == NULL)
			return STATUS_SUCCESS;

		IoQueueWorkItem(pwntcc->WorkItem, PGPWDNTWorkItem,
			CriticalWorkQueue, Context);

#else	// _WIN32_WINNT < 0x5000

		ExInitializeWorkItem(&(pwntcc->WorkItem),PGPWDNTWorkItem,Context);
        ExQueueWorkItem(&(pwntcc->WorkItem),CriticalWorkQueue);

#endif	// _WIN32_WINNT >= 0x5000

        // We have to complete the IRP later
        return STATUS_MORE_PROCESSING_REQUIRED;
    }
	else
	{
        // Now we have to mark Irp as pending if necessary
        if( Irp->PendingReturned )
		{
            IoMarkIrpPending( Irp );
        }

        // Free the completion context
        ExFreePool(pwntcc->FileName);
        ExFreePool(pwntcc);
        return Irp->IoStatus.Status;
    }
}


//----------------------------------------------------------------------
//
// pgpWipeDeleteDispatch
//
// This routine is the main hook routine where we figure out what
// calls are being sent to the file system.
//
//----------------------------------------------------------------------
NTSTATUS
pgpWipeDeleteDispatch (
		PDEVICE_OBJECT HookDevice,
		IN PIRP Irp )
{
    PIO_STACK_LOCATION  currentIrpStack = IoGetCurrentIrpStackLocation(Irp);
    PIO_STACK_LOCATION  nextIrpStack    = IoGetNextIrpStackLocation(Irp);
    PDEVEXTENSION		pdeve;
    PPGPWDNT_COMPLETE_CONTEXT completeContext;
	char				*FileName;
	PVOID				Token;

    // Point at the device extension, which contains information on which
    // file system this IRP is headed for
    pdeve = HookDevice->DeviceExtension;

	// Determine what function we're dealing with
    switch( currentIrpStack->MajorFunction )
	{
	    case IRP_MJ_CLEANUP:
		{
			// Do stuff only if WOD is turned on
			if((tc_flags&WDFLAG_WIPEFILES)!=0)
			{
				// If the file will be deleted, see if we have to PGPWDNT it
				if(currentIrpStack->FileObject->DeletePending &&
				  (SystemProcess != PsGetCurrentProcess())&&
				  (tc_flags!=WDFLAG_NOOP)&&
				  (tc_callback!=0))
				{
					Token=PsReferencePrimaryToken(PsGetCurrentProcess());

					if(Token)
					{
						if(PGPWDNTCheckFileForPGPWDNT(HookDevice,Irp,&FileName))
						{					
							// Tell the OS not to delete the file
							PGPWDNTSetDispositionFile( pdeve->pdevoNext,
								currentIrpStack->FileObject, FALSE );

							// Set up the completion context
							completeContext =
								ExAllocatePool( NonPagedPool, sizeof( PGPWDNT_COMPLETE_CONTEXT));

							completeContext->Irp = Irp;
							completeContext->Token = Token;
							completeContext->DeviceObject = HookDevice;
							completeContext->FileName = FileName;

							IoCopyCurrentIrpStackLocationToNext(Irp);
							IoSetCompletionRoutine( Irp, PGPWDNTHookDone,
								(PVOID) completeContext, TRUE, TRUE, TRUE );

							IoMarkIrpPending( Irp );
							IoCallDriver( pdeve->pdevoNext, Irp );

							// Now stall the IRP
							return STATUS_PENDING;
						}
					}
				}
			}

			// default behavior
			IoSkipCurrentIrpStackLocation(Irp);
			return IoCallDriver( pdeve->pdevoNext, Irp );
		}

	#if	(_WIN32_WINNT >= 0x0500)
		case IRP_MJ_POWER:
			// must use special calls to pass down power IRPs
			PoStartNextPowerIrp(Irp);

			IoSkipCurrentIrpStackLocation(Irp);
			return PoCallDriver(pdeve->pdevoNext, Irp);
	#endif	// _WIN32_WINNT >= 0x5000

		default:
			IoSkipCurrentIrpStackLocation(Irp);
		    return IoCallDriver( pdeve->pdevoNext, Irp );
	}
}


//----------------------------------------------------------------------
//   D R I V E R - E N T R Y   A N D   H O O K   R O U T I N E S
//----------------------------------------------------------------------

//----------------------------------------------------------------------
//
// HookDrive
//
// Hook the drive specified by determining which device object to
// attach to. The algorithm used here is similar to the one used
// internally by NT to determine which device object a file system request
// is directed at.
//
//----------------------------------------------------------------------
BOOLEAN
HookDrive(
		IN USHORT Drive,
		IN PDRIVER_OBJECT DriverObject,
		IN PDRVEXTENSION DriverExtension,
		IN HANDLE VolumeHandle )
{
    PDEVICE_OBJECT      fileSysDevice;
    PDEVICE_OBJECT      hookDevice;
    NTSTATUS            ntStatus;
    PFILE_OBJECT        fileObject;
    IO_STATUS_BLOCK  	ioStatus;
    PDEVEXTENSION		pdeve;
    UCHAR               fsAttribInformation[256];

    //
    // Got the file handle, so now look-up the file-object it refers to
    //
    ntStatus = ObReferenceObjectByHandle( VolumeHandle, FILE_READ_DATA,
                                     NULL, KernelMode, &fileObject, NULL );
    if( !NT_SUCCESS( ntStatus ))
	{
        PGPdbgPrint(("PGPutil: Could not get fileobject from handle: %c\n",
					'A'+Drive ));
        return FALSE;
    }

    //
    // Next, find out what device is associated with the file object by
	// getting its related device object
    //
    fileSysDevice = IoGetRelatedDeviceObject( fileObject );

    if ( ! fileSysDevice )
	{
        PGPdbgPrint(("PGPutil: Could not get related device object: %c\n",
					'A'+Drive ));
        ObDereferenceObject( fileObject );
        return FALSE;
    }

    //
    // The file system's device hasn't been hooked already, so make a
	// hooking device object that will be attached to it.
    //
    ntStatus = IoCreateDevice( DriverObject,
                               sizeof(DEVEXTENSION),
                               NULL,
                               fileSysDevice->DeviceType,
                               0,
                               FALSE,
                               &hookDevice );

    if ( !NT_SUCCESS(ntStatus) )
	{
        PGPdbgPrint(("PGPutil: failed to create associated device: %c\n",
					'A'+Drive ));
        ObDereferenceObject( fileObject );
        return FALSE;
    }

    //
    // Clear the device's init flag as per NT DDK KB article on creating device
    // objects from a dispatch routine
    //
    hookDevice->Flags &= ~DO_DEVICE_INITIALIZING;

    //
    // Setup the device extensions. The drive letter and file system object
	// are stored in the extension.
    //
    pdeve = hookDevice->DeviceExtension;
	pdeve->ulDeviceType			= WIPE_DELETE_DEV;
	pdeve->pdrve				= DriverExtension;
	pdeve->ulSkipCount			= 0;			// not used by this device
    pdeve->ucLogicalDrive		= 'A'+Drive;
    pdeve->pdevoNext			= fileSysDevice;
    pdeve->bIsNTFS				= FALSE;
    pdeve->bIsAnsi				= FALSE;

    //
    // Determine which file system this is. If this fails, assume its NOT NTFS
    //
    ntStatus = ZwQueryVolumeInformationFile(
					VolumeHandle, &ioStatus, fsAttribInformation,
					 sizeof(fsAttribInformation), FileFsAttributeInformation );
    if( !NT_SUCCESS( ntStatus ))
	{
        PGPdbgPrint(("PGPutil: Could not get volume attributes: %x\n", ntStatus));
    }
	else if( RtlCompareMemory(
			((PFILE_FS_ATTRIBUTE_INFORMATION)fsAttribInformation)->FileSystemName,
			L"NTFS", 8 ) == 8)
	{
        pdeve->bIsNTFS = TRUE;
        PGPdbgVerbosePrint(("  => Its an NTFS volume\n"));
    }

    //
    // Finally, attach to the device. The second we're successfully attached,
	// we may start receiving IRPs targetted at the device we've hooked.
    //
    if ( IoAttachDeviceToDeviceStack( hookDevice, fileSysDevice ) == NULL )
	{
        //
        // Couldn't attach for some reason
        //
        PGPdbgPrint(("PGPutil: Connect with Filesystem failed: %c (%x) =>%x\n",
                  'A'+Drive, fileSysDevice, ntStatus ));

        //
        // Derefence the object and get out
        //
        ObDereferenceObject( fileObject );
        return FALSE;
    }

    //
    // Note which device this is
    //
    LDriveDevices[Drive] = hookDevice;

    //
    // Dereference the object
    //
    ObDereferenceObject( fileObject );

    return TRUE;
}


//----------------------------------------------------------------------
//
// Installable driver initialization. Here we just set ourselves up.
// This function is called from the drivers DriverEntry routine.
//
//----------------------------------------------------------------------
NTSTATUS
pgpWipeDeleteInit (
		IN PDRIVER_OBJECT	DriverObject,
		IN PDRVEXTENSION	DriverExtension)
{
    NTSTATUS                ntStatus;
    WCHAR                   volumeBuffer[] = L"\\DosDevices\\X:\\";
    UNICODE_STRING          volumeBufferUnicodeString;
    USHORT                  drive;
    HANDLE 		            volumeHandle;
    OBJECT_ATTRIBUTES       objectAttributes;
    IO_STATUS_BLOCK  	    ioStatus;
    BOOLEAN                 errorMode;
    FILE_FS_DEVICE_INFORMATION fsDeviceInformation;

    PGPdbgVerbosePrint (("PGPutil: entering pgpWipeDeleteInit\n"));

	// Some setup work to initialize the driver
	KeInitializeSemaphore(&apc_sem,0,1);
	num_scheds=0;

    //
    // If not NT 4.0 Final Release, shorten the Fast I/O table so that PGPWDNT
    // will work on the Betas and Release Candidates
    //
    if( *NtBuildNumber < NT4FINAL )
	{
        FastIOHook.SizeOfFastIoDispatch =
				(ULONG) &FastIOHook.FastIoQueryNetworkOpenInfo -
				(ULONG) &FastIOHook;
    }

    //
    // Set up the Fast I/O dispatch table
    //
    DriverObject->FastIoDispatch = &FastIOHook;

    //
    // Disable hard errors so that we don't pop-up when touching removable media
    //
    errorMode = IoSetThreadHardErrorMode( FALSE );

    //
    // Need to check each drive to see if its a local hard disk
    //
    RtlInitUnicodeString (&volumeBufferUnicodeString,
                          volumeBuffer );
    for(drive = 2; drive < 26; drive++ )
	{
        LDriveDevices[drive] = NULL;

        volumeBufferUnicodeString.Buffer[12] = drive+'A';

        InitializeObjectAttributes( &objectAttributes,
									&volumeBufferUnicodeString,
                                    OBJ_CASE_INSENSITIVE, NULL, NULL );

        ntStatus = ZwCreateFile( &volumeHandle, FILE_ANY_ACCESS,
                     &objectAttributes, &ioStatus, NULL, 0,
					 FILE_SHARE_READ|FILE_SHARE_WRITE, FILE_OPEN,
                     FILE_SYNCHRONOUS_IO_NONALERT|
					 FILE_OPEN_FOR_BACKUP_INTENT|FILE_DIRECTORY_FILE,
                     NULL, 0 );
        if( !NT_SUCCESS( ntStatus ) )
		{
            PGPdbgPrint(("PGPutil: Could not open drive %c: %x\n",
						'A'+drive, ntStatus ));

        }
		else
		{
            //
            // Opened the volume. Now go and query it to see if its a local drive
            //
            ntStatus = ZwQueryVolumeInformationFile(
						volumeHandle, &ioStatus, &fsDeviceInformation,
                        sizeof(fsDeviceInformation), FileFsDeviceInformation );

            if( NT_SUCCESS( ntStatus ))
			{
                //
                // OK, see if it's the correct type
                //
                if( fsDeviceInformation.DeviceType == FILE_DEVICE_DISK &&
					!(fsDeviceInformation.Characteristics &
						(FILE_REMOVABLE_MEDIA|FILE_READ_ONLY_DEVICE|
						FILE_FLOPPY_DISKETTE|FILE_WRITE_ONCE_MEDIA|
						FILE_REMOTE_DEVICE|FILE_VIRTUAL_VOLUME)))
				{
                    PGPdbgVerbosePrint(("Volume %C must be filtered\n", 'A'+drive ));

                    //
                    // It is, so hook it.
                    //
					HookDrive( drive, DriverObject,
									DriverExtension, volumeHandle );
                }

                ZwClose( volumeHandle );
            }
        }
    }

    //
    // Save away system process
    //
    SystemProcess = PsGetCurrentProcess();

    IoSetThreadHardErrorMode( errorMode );

///    return ntStatus;
///>>> fix before ship
	return STATUS_SUCCESS;
///<<<
}

//----------------------------------------------------------------------
//
//  called by dispatch routine to process IOCTL
//
//----------------------------------------------------------------------
VOID
pgpWipeDeleteProcessOperation (
    PPGPWIPEDELETESTRUCT	ppwds,
    ULONG					ulStatusFlags)
{
	if (!(ulStatusFlags & kPGPUDFlag_WipeDeleteInitialized))
	{
		ppwds->ulError = kPGPUDError_DriverUninitialized;
		return;
	}

	switch (ppwds->ulOperation)
	{
		case kPGPUDOperation_RegisterCallback :
		{
			// Get the callback
			if(num_scheds!=0)
			{
				ppwds->ulError = kPGPUDError_WipePending;
			}
			else
			{
				tc_callback=(ULONG)ppwds->usCallback;

				ppwds->ulError = kPGPUDError_NoErr;
			}
			break;
		}

		case kPGPUDOperation_RegisterThread :
		{
			// get current thread
			tc_thread=KeGetCurrentThread();

			ppwds->ulError = kPGPUDError_NoErr;
			break;
		}

		case kPGPUDOperation_RegisterFlags :
		{
			// get current thread
			tc_flags=(ULONG)ppwds->ulFlags;

			ppwds->ulError = kPGPUDError_NoErr;
			break;
		}

		case kPGPUDOperation_MapMemory :
		{
			strcpy(ppwds->usFileName,(char *)(ppwds->keFileRecord));

			ppwds->ulError = kPGPUDError_NoErr;
			break;
		}

		case kPGPUDOperation_WipingDone :
		{
			KeReleaseSemaphore (&apc_sem,0,1,FALSE);

			ppwds->ulError = kPGPUDError_NoErr;
			break;
		}

		default :
			ppwds->ulError = kPGPUDError_UndefinedOperation;
			break;
	}
}



/*__Editor_settings____

	Local Variables:
	tab-width: 4
	End:
	vi: ts=4 sw=4
	vim: si
_____________________*/
